(function () {
  'use strict';

  const rootEl = document.documentElement;
  const btnTheme = document.querySelector('[data-action="theme-toggle"]');
  const btnShare = document.querySelector('[data-action="share"]');
  const btnMenu = document.querySelector('[data-action="menu-toggle"]');
  const navPanel = document.getElementById('nav-panel');

  function setTheme(theme) {
    rootEl.setAttribute('data-theme', theme);
    try { localStorage.setItem('theme', theme); } catch (e) {}
    if (btnTheme) {
      btnTheme.setAttribute('aria-label', theme === 'dark' ? 'Switch to light mode' : 'Switch to dark mode');
      btnTheme.setAttribute('aria-pressed', theme === 'dark' ? 'true' : 'false');
    }
  }

  function getPreferredTheme() {
    try {
      const saved = localStorage.getItem('theme');
      if (saved === 'light' || saved === 'dark') return saved;
    } catch (e) {}
    return (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) ? 'dark' : 'light';
  }

  function toast(msg) {
    const el = document.getElementById('toast');
    if (!el) return;
    el.textContent = msg;
    el.classList.add('show');
    window.setTimeout(() => el.classList.remove('show'), 2200);
  }

  function copyToClipboard(text) {
    if (navigator.clipboard && navigator.clipboard.writeText) {
      return navigator.clipboard.writeText(text);
    }
    const ta = document.createElement('textarea');
    ta.value = text;
    ta.setAttribute('readonly', 'readonly');
    ta.style.position = 'fixed';
    ta.style.left = '-9999px';
    document.body.appendChild(ta);
    ta.select();
    try {
      document.execCommand('copy');
      document.body.removeChild(ta);
      return Promise.resolve();
    } catch (e) {
      document.body.removeChild(ta);
      return Promise.reject(e);
    }
  }

  if (btnTheme) {
    btnTheme.addEventListener('click', function () {
      const current = rootEl.getAttribute('data-theme') || 'light';
      setTheme(current === 'dark' ? 'light' : 'dark');
    });
  }

  if (btnShare) {
    btnShare.addEventListener('click', async function () {
      const shareData = {
        title: document.title,
        text: document.querySelector('meta[name="description"]')?.getAttribute('content') || document.title,
        url: window.location.href
      };

      try {
        if (navigator.share) {
          await navigator.share(shareData);
          return;
        }
      } catch (e) {
        // user cancelled or share failed; fallback to copy
      }

      try {
        await copyToClipboard(shareData.url);
        toast('Link copied.');
      } catch (e) {
        toast('Unable to copy link.');
      }
    });
  }

  function setPanel(open) {
    if (!navPanel || !btnMenu) return;
    navPanel.setAttribute('data-open', open ? 'true' : 'false');
    btnMenu.setAttribute('aria-expanded', open ? 'true' : 'false');
    if (open) {
      const firstLink = navPanel.querySelector('a');
      if (firstLink) firstLink.focus();
    } else {
      btnMenu.focus();
    }
  }

  if (btnMenu && navPanel) {
    btnMenu.addEventListener('click', function () {
      const isOpen = navPanel.getAttribute('data-open') === 'true';
      setPanel(!isOpen);
    });

    document.addEventListener('keydown', function (e) {
      if (e.key === 'Escape') {
        setPanel(false);
      }
    });

    navPanel.addEventListener('click', function (e) {
      const t = e.target;
      if (t && t.matches('[data-action="close-nav"]')) {
        e.preventDefault();
        setPanel(false);
      }
    });

    // close panel when clicking a link
    navPanel.querySelectorAll('a').forEach(function (a) {
      a.addEventListener('click', function () {
        setPanel(false);
      });
    });
  }

  setTheme(getPreferredTheme());
})();
